/******************************************************************************
 *                               msniff                                       *
 *                                                                            *
 *                          Leipzig im Sommersemester 2004, Martin Christian. *
 ******************************************************************************/

#include "msniff.h"

/***
 * ToDo:
 *   -> Nutzereingaben nicht ungeparst benutzen: options->proto
 *   -> exit-Handler registrieren
 *   -> 
 ***/

/* Variablen aus der getopt Bibliothek */
extern char *optarg;
extern int optind, opterr, optopt;

const char options_def[] = "s:M:t:m:o:p:wh";
struct options_t *options;

/* ************************ USAGE ******************************************* */
void usage( char* cmd ) {
	printf( "Usage:\t%s -p protocol interface\n", cmd );
	printf( "\t%s -s sender_ip [-M sender_mac] -t target_ip [-m target_mac]\n", cmd );
	printf( "\t\t -o operation -p protocol interface\n" );
	printf( "\t%s -w interface\n", cmd );
	printf( "\t%s -h\n", cmd );
}
/* ************************************************************************** */

/* ************************ SCANIP ****************************************** */
uint32_t scanip( const char *ipstring ) {
	uint32_t res;
	uint8_t *addr;
	int ipvec[4];
	int i;

	addr = (uint8_t*)&res;
	i = sscanf( ipstring, "%d.%d.%d.%d", &ipvec[0], &ipvec[1], &ipvec[2], &ipvec[3] );
	if ( i < 4 ) return -1;

	for( i=0; i<4; i++ )
		addr[i] = ipvec[i];

	return res;
}
/* ************************************************************************** */

/* ************************ SCANMAC ***************************************** */
struct libnet_ether_addr* scanmac( const char *macstring ) {
	struct libnet_ether_addr* macaddr;
	int macvec[6];
	int n;

	n = sscanf( macstring, "%x:%x:%x:%x:%x:%x",
							&macvec[0], &macvec[1], &macvec[2],
							&macvec[3], &macvec[4], &macvec[5] );
	if ( n != 6 ) return NULL;

	macaddr = (struct libnet_ether_addr*)malloc( sizeof(struct libnet_ether_addr) );
	if ( macaddr == NULL ) {
		printf( "scanmac(): Couldn't allocate memory for macaddr!\n" );
		return NULL;
	}

	for ( n=0; n<6; n++ ) macaddr->ether_addr_octet[n] = macvec[n];
	return macaddr;
}
/* ************************************************************************** */

/* ************************ MACCMP ****************************************** */
int maccmp( struct libnet_ether_addr *ea0, struct libnet_ether_addr *ea1 ) {
	int i;
	for ( i=0; i<6; i++ ) {
		if ( ea0->ether_addr_octet[i] != ea1->ether_addr_octet[i] )
			return -1;
	}
	return 0;
}
/* ************************************************************************** */

/* ************************ CLEANUP ***************************************** */
void cleanup( libnet_t *l ) {
	int i, n;

	if ( options != NULL ) {
		if ( options->target_mac != NULL ) {
			free( options->target_mac );
		}
		free( options );
		/*
		if ( options->sender_mac != NULL ) {
			free( options->sender_mac );
		}
		free( options );
		*/
	}
	if ( l != NULL ) libnet_destroy( l );

	/* cleanup all captured ARP entries before cleaning list */
	LISTFRAME *lf;
	struct arp_watch_info *info;

	lf = list_top();
	if ( lf != NULL ) {
		info = (struct arp_watch_info*)lf->content;
		n = list_get_count();
		for ( i=0; i<n; i++ ) free( info );
	}
	list_destroy();
}
/* ************************************************************************** */

/* ************************ INIT ******************************************** */
struct options_t* init( int argc, char** argv ) {
	struct options_t *options;
	int i;

	options = (struct options_t*)malloc( sizeof(struct options_t) );

	/* init options */
	options->dev[0] = '\0';
	options->target_mac = NULL;
	options->sender_mac = NULL;
	options->target_ip = -1;
	options->sender_ip = -1;
	options->proto[0] = '\0';
	options->op = 0;
	options->action = ACTION_SNIFFING;

	/* parsing options */
	i = getopt( argc, argv, options_def );
	while( i != -1 ) {
		switch( i ) {
		case 's' : /* sender ip */
			if ( options->action != ACTION_SNIFFING ) {
				printf( "init(): Conflicting options have been used!\n" );
				usage( argv[0] );
				cleanup( NULL );
				return NULL;
			}

			options->action = ACTION_SPOOFING;
			options->sender_ip = scanip( optarg );
			if ( options->sender_ip == -1 ) {
				printf( "init(): Sender IP has wrong format.\n" );
				cleanup( NULL );
				return NULL;
			}
			break;
		case 'M' : /* sender mac */
			options->sender_mac = scanmac( optarg );
			if ( options->sender_mac == NULL ) {
				printf( "init(): Sender MAC address has wrong format.\n" );
				cleanup( NULL );
				return NULL;
			}
			break;
		case 't' : /* target ip */
			options->target_ip = scanip( optarg );
			if ( options->target_ip == -1 ) {
				printf( "init(): Target IP has wrong format.\n" );
				cleanup( NULL );
				return NULL;
			}
			break;
		case 'm' : /* target mac */
			options->target_mac = scanmac( optarg );
			if ( options->target_mac == NULL ) {
				printf( "init(): Target MAC address has wrong format.\n" );
				cleanup( NULL );
				return NULL;
			}
			break;
		case 'o' : /* operation */
			i = sscanf( optarg, "%hd", &options->op );
			if ( i != 1 ) {
				printf( "init(): Error reading operation parameter.\n" );
				cleanup( NULL );
				return NULL;
			}
			break;
		case 'p' : /* protocol */
			strncpy( options->proto, optarg, SHORTBUF );
			break;
		case 'w' : /* watching */
			if ( options->action != ACTION_SNIFFING ) {
				printf( "init(): Conflicting options have been used!\n" );
				usage( argv[0] );
				cleanup( NULL );
				return NULL;
			}
			options->action = ACTION_WATCHING;
			break;
		case 'h':
			usage( argv[0] );
			cleanup( NULL );
			exit( EXIT_SUCCESS );
		default :
			usage( argv[0] );
			cleanup( NULL );
			return NULL;
		}
		i = getopt( argc, argv, options_def );
	}

	argc -= optind;
	if ( argc != 1 ) {
		usage( argv[0] );
		cleanup( NULL );
		return NULL;
	}
	argv += optind;
	strcpy( options->dev, argv[0] );

	return options;
}
/* ************************************************************************** */

/* ************************ ANALYZE_ARP ************************************* */
void analyze_arp( const struct arp_t *arphdr ) {
	#ifdef DEBUG
		printf( "=== ARP Data ===\n" );
		printf( "Hardware-Address-Type: %d\n", ntohs(arphdr->ar_hrd) );
		printf( "Protocol-Type: 0x%x\n", ntohs(arphdr->ar_pro) );
		printf( "Hardware-Address-Size: %d\n", arphdr->ar_hln );
		printf( "Prtotocol-Address-Size: %d\n", arphdr->ar_pln );
		printf( "Operation: %d\n", ntohs(arphdr->ar_op) );
	#endif

	switch ( ntohs(arphdr->ar_op) ) {
	case ARPOP_REQUEST :
		printf( "From %d.%d.%d.%d (%x:%x:%x:%x:%x:%x): who has %d.%d.%d.%d\n\n",
						arphdr->ar_sip[0], arphdr->ar_sip[1], arphdr->ar_sip[2],
						arphdr->ar_sip[3],
						arphdr->ar_sha[0], arphdr->ar_sha[1], arphdr->ar_sha[2],
						arphdr->ar_sha[3], arphdr->ar_sha[4], arphdr->ar_sha[5],
						arphdr->ar_tip[0], arphdr->ar_tip[1], arphdr->ar_tip[2],
						arphdr->ar_tip[3] );
		break;
	case ARPOP_REPLY :
		printf( "To %d.%d.%d.%d (%x:%x:%x:%x:%x:%x): %d.%d.%d.%d is at %x:%x:%x:%x:%x:%x\n\n",
						arphdr->ar_tip[0], arphdr->ar_tip[1], arphdr->ar_tip[2],
						arphdr->ar_tip[3],
						arphdr->ar_tha[0], arphdr->ar_tha[1], arphdr->ar_tha[2],
						arphdr->ar_tha[3], arphdr->ar_tha[4], arphdr->ar_tha[5],
						arphdr->ar_sip[0], arphdr->ar_sip[1], arphdr->ar_sip[2],
						arphdr->ar_sip[3],
						arphdr->ar_sha[0], arphdr->ar_sha[1], arphdr->ar_sha[2],
						arphdr->ar_sha[3], arphdr->ar_sha[4], arphdr->ar_sha[5] );
		break;
	case ARPOP_REVREQUEST : break;
	case ARPOP_REVREPLY : break;
	default : printf( "Operation is unknown or not implemented!\n\n" );
	}
}
/* ************************************************************************** */

/* ************************ANALYZE_TCP ************************************** */
void analyze_tcp( const struct tcp_t *tcphdr ) {
	printf( "source_port=%d | destination_port=%d | seqno=%ud | ackno=%ud | winsize=%d\n\n",
					ntohs(tcphdr->tcp_src_port), ntohs(tcphdr->tcp_dst_port),
					ntohl(tcphdr->tcp_seqnr), ntohl(tcphdr->tcp_ack),
					ntohs(tcphdr->tcp_winsize) );
}
/* ************************************************************************** */

/* ************************ ANALYZE_UDP ************************************* */
void analyze_udp( const struct udp_t *udphdr ) {
	printf( "source_port=%d | destination_port=%d | payload_size=%d\n\n",
					ntohs(udphdr->udp_src_port), ntohs(udphdr->udp_dst_port),
					ntohs(udphdr->udp_len) );
}
/* ************************************************************************** */

/* ************************ ANALYZE_IP6 ************************************* */
void analyze_ip6( const struct ip6_t *iphdr ) {
}
/* ************************************************************************** */

/* ************************ ANALYZE_IP ************************************** */
void analyze_ip( const struct ip4_t *iphdr ) {
	void *nexthdr;

	if ( iphdr->ip_vers == 4 ) {
		nexthdr = (void*)( (uint32_t*)iphdr + iphdr->ip_hdrlen );
		printf( "IP: Source=%d.%d.%d.%d | Destination=%d.%d.%d.%d | TTL=%d | Size=%d\n",
						iphdr->ip_src_addr[0], iphdr->ip_src_addr[1], iphdr->ip_src_addr[2],
						iphdr->ip_src_addr[3],
						iphdr->ip_dst_addr[0], iphdr->ip_dst_addr[1], iphdr->ip_dst_addr[2],
						iphdr->ip_dst_addr[3],
						iphdr->ip_ttl, ntohs(iphdr->ip_total_len) );

		switch( iphdr->ip_proto ) {
			case IPPROTO_TCP:
				printf( "TCP: " );
				analyze_tcp( nexthdr );
				break;
			case IPPROTO_UDP:
				printf( "UDP: " );
				analyze_udp( nexthdr );
				break;
			case IPPROTO_IPV6:
				printf( "IPv6: " );
				analyze_ip6( nexthdr );
				break;
			default: printf( "Unknown packet.\n\n" );
		}
	}
	else if ( iphdr->ip_vers == 6 )
		analyze_ip6( (struct ip6_t*)iphdr );
	else printf( "Unknown IP version: %d!\n", iphdr->ip_vers );
}
/* ************************************************************************** */

/* ************************ SNIFF_PACKET ************************************ */
void sniff_packet( u_char* args, const struct pcap_pkthdr *header,
									 const u_char* data ) {

	struct ether_t *etherhdr = (struct ether_t*)data;
	struct vether_t *vetherhdr = (struct vether_t*)data;

	printf( "%d of %d bytes captured at %s",
					header->caplen, header->len,
					ctime( &(header->ts.tv_sec) )	);

	switch ( ntohs(etherhdr->ether_type) ) {
	case 0x0800 :
		analyze_ip( (struct ip4_t*)(etherhdr->ether_data) );
		break;
	case 0x0806 :
		analyze_arp( (struct arp_t*)(etherhdr->ether_data) );
		break;
	case 0x8100 :
		printf( "Found ethernet VLAN packet!\n" );
		switch ( ntohs(vetherhdr->ether_type) ) {
		case 0x0800 :
			analyze_ip( (struct ip4_t*)(etherhdr->ether_data) );
			break;
		case 0x0806 :
			analyze_arp( (struct arp_t*)(vetherhdr->ether_data) );
			break;
		default:
			printf( "Unknown Protocol!\n" );
		}
		break;
	default:
		printf( "Unknown Protocol!\n" );
	}
}
/* ************************************************************************** */

/* ************************ WATCH_ARP *************************************** */
void watch_arp( u_char* args, const struct pcap_pkthdr *header, const u_char* data ) {

	struct ether_t *etherhdr = (struct ether_t*)data;
	struct arp_t *arphdr;
	struct arp_watch_info *info, *li;
	int i, n;

	if ( ntohs(etherhdr->ether_type) != 0x0806 ) return;

	arphdr = (struct arp_t*)(etherhdr->ether_data);
	info = (struct arp_watch_info*)malloc( sizeof(struct arp_watch_info) );

	analyze_arp( arphdr );

	info->time = header->ts.tv_sec;
	info->ipaddr = ntohl( *((uint32_t*)arphdr->ar_sip) );
	memcpy( info->macaddr.ether_addr_octet, arphdr->ar_sha, 6 );

	n = list_get_count();
	for ( i=0; i<n; i++ ) {
		li = (struct arp_watch_info*)list_get( i );
		printf( "%x   %x:%x:%x:%x:%x:%x   %s\n",
						li->ipaddr, li->macaddr.ether_addr_octet[0], li->macaddr.ether_addr_octet[1],
						li->macaddr.ether_addr_octet[2], li->macaddr.ether_addr_octet[3],
						li->macaddr.ether_addr_octet[4], li->macaddr.ether_addr_octet[5],
						ctime(&(li->time)) );

		if ( li->ipaddr == info->ipaddr ) {
			if ( difftime(info->time, li->time) < 30 ) {
				if ( maccmp(&info->macaddr, &li->macaddr) != 0 ) {
					printf( "ARP spoofing detected!\n" );
					free( info );
				}
				else {
					li->time = info->time;
					free( info );
				}
			}
			else {
				/* replace old entry */
				memcpy( li->macaddr.ether_addr_octet, info->macaddr.ether_addr_octet, 6 );
				li->time = info->time;
				free( info );
			}
			break;
		}
	}
	if ( i == n ) list_insert( info );
}
/* ************************************************************************** */

/* ************************ SEND_ARP **************************************** */
int send_arp( libnet_t *libnetcon ) {
	int i;

	switch( options->op ) {
		case ARPOP_REQUEST :
			if ( options->target_mac == NULL ) {
				i = sizeof(struct libnet_ether_addr);
				options->target_mac = (struct libnet_ether_addr*)malloc( i );
				memcpy( options->target_mac, &ether_broadcast, i );
			}
			printf( "Sending ARP Request on %s...", options->dev );
			break;
		case ARPOP_REPLY:
			if ( options->target_mac == NULL ) {
				printf( "Can't send ARP reply, because MAC address is missing!\n" );
				cleanup( libnetcon );
				exit( EXIT_FAILURE );
			}
			printf( "Sending ARP Reply on %s...", options->dev );
			break;
		default:
			options->op = ARPOP_REQUEST;
	}
	/* Signatur:
			libnet_build_arp( hw type (ether=0x01), proto type (arp fuer ip=0x0800),
												hwaddr size, protoaddr size, optype,
												sender hwaddr, sender protoaddr,
												target hwaddr, target protoaddr,
												payload, payload size, libnet context, ptag )
	*/
	i = libnet_build_arp( ARPHRD_ETHER, ETHERTYPE_IP, 6, 4, options->op,
												options->sender_mac->ether_addr_octet,
												(uint8_t*)&(options->sender_ip),
												options->target_mac->ether_addr_octet,
												(uint8_t*)&(options->target_ip),
												NULL, 0, libnetcon, 0 );
	if ( i == -1 ) {
		printf( "error!\nCan't build ARP header: %s\n", libnet_geterror(libnetcon) );
		return -1;
	}

	/*
	 *
	 * libnet_build_ethernet( ether_dest, ether_src, proto_type,
	 * 												payload, payload size, libnet handle, ptag )
	 */
	i = libnet_build_ethernet( options->target_mac->ether_addr_octet,
							options->sender_mac->ether_addr_octet,
							ETHERTYPE_ARP, NULL, 0, libnetcon, 0 );

	if ( i == -1 ) {
		printf( "error!\nCan't build ethernet header: %s\n", libnet_geterror(libnetcon) );
		return -1;
	}

	i = libnet_write( libnetcon );
	if ( i == -1 ) printf( "error!\nWrite error: %s\n", libnet_geterror(libnetcon) );
	else printf( "wrote %d byte ARP packet from context\n", i );

	return i;
}
/* ************************************************************************** */

/* ************************ SEND_PACKET ************************************* */
int send_packet( libnet_t *libnetcon ) {
	int n;

	if ( options->sender_mac == NULL ) {
		options->sender_mac = libnet_get_hwaddr( libnetcon );
		if ( options->sender_mac == NULL ) {
			printf( "init(): %s\n",  libnet_geterror(libnetcon) );
			return -1;
		}
	}
	if ( options->target_ip == -1 )
		options->target_ip = ip_broadcast;
	if ( options->sender_ip == -1 ) {
		options->sender_ip = libnet_get_ipaddr4( libnetcon );
		if ( options->sender_ip == -1 ) {
			printf( "init(): %s\n", libnet_geterror(libnetcon) );
			return -1;
		}
	}

	if ( !strcmp( options->proto, "arp" ) ) {
		n = send_arp( libnetcon );
		if ( n == -1 ) {
			printf( "send_packet(): %s\n", libnet_geterror(libnetcon) );
			return EXIT_FAILURE;
		}
	}
	else if ( !strcmp( options->proto, "ip" ) ) {
		return EXIT_FAILURE;
	}
	else {
		printf( "Unbekanntes oder nicht impelmentiertes Protokoll: %s!\n",
						options->proto );
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
/* ************************************************************************** */

/* ************************ MAIN ******************************************** */
int main( int argc, char** argv ) {
	/* variables related to PCAP */
	char error[PCAP_ERRBUF_SIZE];
	pcap_t *handle;

	/* variables related to libnet */
	libnet_t *libnetcon = NULL;

	/* common variables */
	uint32_t ipaddr, netmask;
	struct bpf_program filter;
	int i;
	int ret = EXIT_SUCCESS;

	options = init( argc, argv );
	if ( options == NULL ) {
		return EXIT_FAILURE;
	}

	if ( (options->action == ACTION_SNIFFING) || (options->action == ACTION_WATCHING) ) {
		void (*callbackfnct)(u_char*, const struct pcap_pkthdr*, const u_char*);
		/* sniffing or watching*/

		/* get interface ip and netmask */
		error[0] = '\0';
		pcap_lookupnet( options->dev, &ipaddr, &netmask, error );
		if ( strlen(error) > 0 ) {
			printf( "%s\n", error );
			cleanup( NULL );
			exit( EXIT_FAILURE );
		}

		/* open sniffing session */
		error[0] = '\0';
		handle = pcap_open_live( options->dev, BUFSIZ, 1, 0, error );
		if ( strlen(error) > 0 ) {
			printf( "%s\n", error );
			cleanup( NULL );
			exit( EXIT_FAILURE );
		}

		if ( options->action == ACTION_SNIFFING ) {
			callbackfnct = sniff_packet;
		}
		else {
			strcpy( options->proto, "arp" );
			callbackfnct = watch_arp;
			list_init();
		}

		/* compile filter expression */
		pcap_compile( handle, &filter, options->proto, 0, netmask );
		/* apply filter to sniffing session */
		pcap_setfilter( handle, &filter );
		/* start sniffing until n packets read */
		printf( "Listening on Interface: %s\n", options->dev );
		pcap_loop( handle, 4, callbackfnct, NULL );

		pcap_close( handle );
	}
	else {
		/* spoofing */
		libnetcon = libnet_init( LIBNET_LINK, options->dev, error );
		if ( libnetcon == NULL ) {
			printf( "libnet_init() failed: %s\n", error );
			cleanup( NULL );
			return EXIT_FAILURE;
    }

		i = send_packet( libnetcon );
		if ( i == -1 ) ret = EXIT_FAILURE;
	}

	cleanup( libnetcon );
	return ret;
}
/* ************************************************************************** */
